/*
 * hp100.c
 * Copyright (c) 1996 Simon Thornington, All Rights Reserved.
 *
 * Ethernet driver: handles HP100VG 2585B PCI cards
 * Derived (very loosely) from the Linux driver.
 *
 * TO DO:
 *        - get it working
 *        - clean it up
 *        - get all the tabs the same
 *        - implement bus mastering???
 *
 */
#include <OS.h>
#include <KernelExport.h>
#include <Application.h>
#include <Drivers.h>
#include "hp100.h"
#include <ether_driver.h>
#include <netconfig.h>
#include <device/PCI.h>
#include <stdarg.h>


static int noprintf(const char *fmt, ...) { return (1); }

/*#define ddprintf ddprintf*/
#define d0printf dprintf	/* errors */
#define d1printf dprintf	/* notifications */
#define d2printf noprintf	/* writer stuff */
#define d3printf noprintf	/* reader stuff */
#define d4printf noprintf	/* interrupt stuff */
#define d5printf noprintf	/* verbose stuff */
#define dtprintf noprintf	/* test statements */

#define DEBUG

#define HP100_MAX_PACKET_SIZE	(1536+4)
#define HP100_MIN_PACKET_SIZE	60

#ifndef HP100_DEFAULT_RX_RATIO
/* default - 65% onboard memory on the card are used for RX packets */
#define HP100_DEFAULT_RX_RATIO	65
#endif

#ifndef HP100_DEFAULT_PRIORITY_TX
/* default - don't enable transmit outgoing packets as priority */
#define HP100_DEFAULT_PRIORITY_TX 0
#endif

#ifndef HP100_IO_MAPPED
#define HP100_IO_MAPPED
#endif

#define HP100_TRANSMIT_TIMEOUT 1000000.0	/* one second */


/*
 * We only care about these interrupts in our driver
 */
#define INTS_WE_CARE_ABOUT (HP100_RX_PACKET | HP100_RX_ERROR | \
							HP100_TX_ERROR | HP100_TX_COMPLETE)

/* 
 * Our private data structure - probably a bit verbose
 */
typedef struct hp100_private {
	unsigned char irq_level;		/* our ISA IRQ level */
	short port;						/* our ISA I/O port number */
	unsigned long mem;				/* our ISA mem address */
	int boundary;					/* boundary register value (mirrored) */
	ether_address_t dev_addr;		/* my ethernet address */
	volatile long iolock;			/* ethercard io, excl interrupt handler */
	volatile int intrlock;			/* ethercard io, incl interrupt handler */
	volatile int interrupted;  		/* interrupted system call */
	volatile long inrw;				/* in read or write function */
	volatile long inint;			/* in interrupt handler */
	volatile long ilock;			/* waiting for input */
	volatile long olock; 			/* waiting to output */
	volatile long rxunack;			/* number of packets unacknowledged */
	volatile cpu_status cps;		/* for intr_lock/intr_unlock */
	long base_addr;
	char name[30];
    u_short soft_model;
    u_int memory_size;
    u_short rx_ratio;				/* 1 - 99 */
    u_short priority_tx;			/* != 0 - priority tx */
    short mem_mapped;				/* memory mapped access */
    u_char *mem_ptr_virt;			/* virtual memory mapped area, maybe NULL */
    u_char *mem_ptr_phys;			/* physical memory mapped area */
    short lan_type;		   			/* 10Mb/s, 100Mb/s or -1 (error) */
    int hub_status;		   			/* login to hub was successful? */
    u_char mac1_mode;
    u_char mac2_mode;

	/* various soft constants */
	short start_rx, end_rx, start_tx, end_tx; /* ring buffer boundaries */

	/*
	 * Various statistics
	 */
	volatile ints;					/* total number of interrupts */
	volatile rints;					/* read interrupts */
	volatile wints;					/* write interrupts */
	volatile reads;					/* reads */
	volatile writes;				/* writes */
	volatile resets;				/* resets */
	volatile rerrs;					/* read errors */
	volatile werrs;					/* write errors */
	volatile interrs;				/* unknown interrupts */
	volatile frame_errs;			/* frame alignment errors */
	volatile crc_errs;				/* crc errors */
	volatile frames_lost;			/* frames lost due to buffer problems */

	/*
	 * Most recent values of the error statistics 
	 */
	int rerrs_last;	
	int werrs_last;	
	int interrs_last;
	int frame_errs_last;
	int crc_errs_last;
	int frames_lost_last;
} hp_private_t;


/*
 * functions to swap shorts and longs 
 */
static inline 
ushort SWAPSHORT(ushort x)
{
    return ((x << 8) | (x >> 8));
}
static inline 
ulong SWAPLONG(ulong x)
{
    return ((SWAPSHORT(x) << 16) | (SWAPSHORT(x >> 16)));
}


/**************************************************************************
* Standard input/output functions to access io regsters                   *
**************************************************************************/

/* 
 * Write a value in a 8 bits register at a specified address 
 */
static void	
outb(uchar value, long address)
{
	*(vuchar *)address = value;
}

/* 
 * Write a value in a 16 bits register at a specified address
 * (respecting intel byte order).
 */
static void	
outw(ushort value, long address)
{
	ushort	tmp;
	tmp = ((value & 0xff) << 8) | ((value >> 8) & 0xff);
	*(vushort *)address = tmp;
}

/* 
 * Write a value in a 32 bits register at a specified address.
 * (don't respect intel byte order, only for memory to register copy)
 */
static void	
outl(ulong value, long address)
{
	*(vulong *)address = value;
}

/*
 * Read the current value of a 8 bits register at a specified address
 */
static uchar	
inb(long address)
{
	return(*(vuchar *)address);
}

/*
 * Read the current value of a 16 bits register at a specified adress
 * (respecting intel byte order).
 */
static ushort	
inw(long address)
{
	ushort	value;
	value = *(vushort *)address;
	return(((value & 0xff) << 8) | ((value >> 8) & 0xff));
}

/*
 * read a long
 */
static ulong	
inl(long address)
{
	ulong value;
	
	value = *(vulong *)address;
	return(value);
}

/*
 * function repeatedly gets longs from the dataport
 * depends on alignment to next long boundary of dest!
 */
static void
hp100_insl(hp_private_t *data,
		   unsigned dataport,
		   unsigned long *dest,
		   unsigned len) 
{
	int ioaddr=data->base_addr;
	register int i;
	
	for (i=0;i<len;i++)
		*dest++=inl( ioaddr+dataport );
}

/*
 * function repeatedly puts longs to the dataport
 * depends on alignment to next long boundary of src!
 */
static void
hp100_outsl(hp_private_t *data,
			unsigned dataport,
			const unsigned long *src,
			unsigned len) 
{
	int ioaddr=data->base_addr;
	register int i;
	
	for (i=0;i<len;i++)
		outl( *src++, ioaddr+dataport );
}


/* should really include all variants, but it's my driver, right?  */
#define HP_VENDOR_ID 	0x103c
#define HP_2585B_ID 	0x1031

int hp100_rx_ratio = HP100_DEFAULT_RX_RATIO;
int hp100_priority_tx = HP100_DEFAULT_PRIORITY_TX;

/*
 *  prototypes (are a pain in the ass)
 */

static long 
hp100_open(device_info *info, ulong flags);
static long 
hp100_close(device_info *info);
static long 
hp100_read(device_info *info, void *buf, ulong buflen, ulong pos);
static long 
hp100_write(device_info *info, void *buf, ulong buflen, ulong pos);
static long 
hp100_control(device_info *info, ulong op, void *buff); 

static bool 
hp100_interrupt( void *data );
static int 
hp100_init(device_info *drvr); 
static int 
hp100_probe(hp_private_t *);
static int 
hp100_probe1(hp_private_t *data, int ioaddr);
static void 
hp100_start_interface( hp_private_t *data );
static void 
hp100_stop_interface( hp_private_t *data );
static void 
hp100_load_eeprom( hp_private_t *data );
static int 
hp100_sense_lan( hp_private_t *dev );
static int 
hp100_login_to_vg_hub( hp_private_t *data );
static int 
hp100_down_vg_link( hp_private_t *data );
static void 
hp100_hwinit(hp_private_t *data);
static int 
copy_packet(hp_private_t *data, 
					   unsigned char *hp100_buf, 
					   int buflen);


long
inrw(hp_private_t *data)
{
	return (data->inrw);
}

/*
 * Global variable for our data struct - not very good for multiple NICS!
 */
static hp_private_t hp100_data;

/*
 * io_lock gets you exclusive access to the card, except that 
 * the interrupt handler can still run.
 */
#define io_lock(data)			acquire_sem(data->iolock)
#define io_unlock(data)			release_sem(data->iolock)

/*
 * output_wait wakes up when the card is ready to transmit another packet
 */
#define output_wait(data, t)	acquire_sem_etc(data->olock, 1, B_TIMEOUT, t)
#define output_unwait(data, c)	release_sem_etc(data->olock, c, B_DO_NOT_RESCHEDULE)

/*
 * input_wait wakes up when the card has at least one packet on it
 */
#define input_wait(data)		acquire_sem(data->ilock)
#define input_unwait(data, c)	release_sem_etc(data->ilock, c, B_DO_NOT_RESCHEDULE)


/*
 * How many waiting for input?
 */
static long
input_count(hp_private_t *data)
{
	long count;

	get_sem_count(data->ilock, &count);
	return (count);
}


/*
 * Spinlock for negotiating access to card with interrupt handler
 */

static void inline
intr_lock(hp_private_t *data)
{
	data->cps=disable_interrupts();
	acquire_spinlock((volatile long *)&data->intrlock);
}

static void inline
intr_unlock(hp_private_t *data)
{
	release_spinlock((volatile long *)&data->intrlock);
	restore_interrupts(data->cps);
}

/*
 * The interrupt handler must lock all calls to the card
 * This macro is useful for that purpose.
 */
#define INTR_LOCK(data, expression) (intr_lock(data), (expression), intr_unlock(data))

/* 
 * bring down the VG link and un-login
 * what is the -2 for?
 */
static int 
hp100_down_vg_link( hp_private_t *data )
{
	int ioaddr = data -> base_addr;
	int i;

	hp100_page( MAC_CTRL );
	for ( i = 2500; i > 0; i-- )
		if ( hp100_inb( VG_LAN_CFG_1 ) & HP100_LINK_CABLE_ST ) break;
	if ( i <= 0 )				/* not signal - not logout */
	return 0;
	hp100_andb( ~HP100_LINK_CMD, VG_LAN_CFG_1 );
	for (i=2500;i>0;i++)
		if ( !( hp100_inb( VG_LAN_CFG_1 ) & ( HP100_LINK_UP_ST | 
											  HP100_LINK_CABLE_ST | 
											  HP100_LINK_GOOD_ST ) ) )
			return 0;
	d0printf( "hp100_down_vg_link: timeout\n" );
	return -2;
}

/* 
 * Establish connection with hub 
 */
static int 
hp100_login_to_vg_hub( hp_private_t *data )
{
	int i;
	int ioaddr = data -> base_addr;
	u_short val=0;

	hp100_page( MAC_CTRL );
	/* bug fix re: new cards, old hubs */
	hp100_outb( 0x00, VG_LAN_CFG_2);
	/* ask for nothing - prohibits promiscuous mode, desires 802.3 frames,
	   tell hub not a repeater */
	hp100_outw( 0x0000, TRAIN_REQUEST);
	
	hp100_orb( HP100_LINK_CMD|HP100_LOAD_ADDR|HP100_VG_RESET, VG_LAN_CFG_1 );
	for (i=5000;i>0;i--)
		if ( hp100_inb( VG_LAN_CFG_1 ) & HP100_LINK_CABLE_ST ) break;
	if (i <= 0) {
		return B_ERROR;
	}
	
	for (i=2500;i>0;i--) {
		val = hp100_inb( VG_LAN_CFG_1 );
		if ( ( val & ( HP100_LINK_UP_ST | HP100_LINK_GOOD_ST ) ) == 
                 	 ( HP100_LINK_UP_ST | HP100_LINK_GOOD_ST ) )
		return B_NO_ERROR;	/* success */
	}
	if ( val & HP100_LINK_GOOD_ST )
		d0printf( "hp100: 100Mb cable training failed, check cable.\n");
	else
		d0printf( "hp100: 100Mb node not accepted by hub, check frame type or security.\n");

	hp100_down_vg_link( data );
	hp100_page( MAC_CTRL );
	hp100_andb( ~HP100_LOAD_ADDR, VG_LAN_CFG_1 );
	hp100_orb( HP100_LINK_CMD, VG_LAN_CFG_1 );
	return B_ERROR;
}

/* 
 * Detect the type of LAN the adapter is attached to.
 * Hasn't actually been tested with a 10Mbit 802.3...
 * return values: LAN_10, LAN_100 or LAN_ERR (not connected or hub is down)... 
 */
static int 
hp100_sense_lan( hp_private_t *dev )
{
	int i;
	int ioaddr = dev -> base_addr;
	u_short val_VG, val_10;
	
	hp100_page( MAC_CTRL );
	/* sensing means a bit of training, better poke the bug just in case */
	hp100_outb( 0x00, VG_LAN_CFG_2 );
	hp100_orb( HP100_LINK_CMD|HP100_LOAD_ADDR|HP100_VG_RESET, VG_LAN_CFG_1);

	val_10 = hp100_inb( 10_LAN_CFG_1 );
	val_VG = hp100_inb( VG_LAN_CFG_1 );
	if ( val_10 & HP100_LINK_BEAT_ST ) return HP100_LAN_10;
	for ( i = 0; i < 5000; i++ ) {
		val_VG = hp100_inb( VG_LAN_CFG_1 );
		if ( val_VG & HP100_LINK_CABLE_ST ) return HP100_LAN_100;
	}
	return HP100_LAN_ERR;
}


/*
 *  preliminary PCI probe
 */
int 
hp100_probe( hp_private_t *data)
{
	int base_addr = 0;
	long int ioaddr,tst;
	int pci_start_index = 0;
	pci_info pci;
	int pci_index;
      
	for ( pci_index = 0; pci_index < 8; pci_index++ )
	{        
		tst=get_nth_pci_info( pci_index, &pci);
		if (tst == B_NO_ERROR) {
			if ((pci.vendor_id == HP_VENDOR_ID) &&
				(pci.device_id == HP_2585B_ID)) 
					break;
		}
		else return(B_ERROR);
	}

	ioaddr=pci.u.h0.base_registers[0];
	ioaddr &= ~3;		/* remove space marker */

	/* this is a pci style interrupt identifier! */
    data->irq_level = pci.u.h0.interrupt_line;
	return (hp100_probe1( data, ioaddr ));
}

/* 
 *  more detailed probe, including some initialization
 */
static int 
hp100_probe1( hp_private_t *data, int ioaddr )
{
	int i;
	u_int uc;
	u_char *macbyte;

	data -> base_addr = ioaddr;
	strcpy(data -> name, "HP 100VG 2585B PCI");

	hp100_page( HW_MAP );
	if ( data -> irq_level == 2 ) data -> irq_level = 9;
	data -> memory_size = (8192 << ( ( hp100_inb( SRAM ) >> 5 ) &0x07 ));
	data -> rx_ratio = hp100_rx_ratio;

	/* turn off bus mastering - necessary, for now ;^) */
	hp100_outw( (HP100_BM_WRITE | HP100_BM_READ | HP100_RESET_HB), OPTION_LSW );

	hp100_page( ID_MAC_ADDR );
	data -> soft_model = hp100_inb( SOFT_MODEL );
	data -> mac1_mode = HP100_MAC1MODE3;
	data -> mac2_mode = HP100_MAC2MODE3;
  	d1printf("hp100: MAC Addr = ");
	macbyte=&(data->dev_addr).ebyte[0];
	for ( i = uc = 0; i < 6; i++ ) {
		macbyte[i] = hp100_inb( LAN_ADDR + i );
		d1printf("%02x",macbyte[i]);
		if (i<5) d1printf(":");
		else d1printf("\n");
	}
	d1printf("%s at 0x%x, IRQ %d, PCI", data -> name, ioaddr, 
										(data -> irq_level & 0x0f) );
	d1printf(" bus, %dk SRAM (rx/tx %d%%).\n", data -> memory_size >> 10, 
											   data -> rx_ratio );
	d1printf("%s: ", data -> name );
	/* needed to sense the LAN type correctly */ 
	hp100_hwinit(data);
	data -> lan_type = hp100_sense_lan( data );
	if ( data -> lan_type != HP100_LAN_ERR )
		d1printf("Adapter is attached to " );
	switch ( data -> lan_type ) {
		case HP100_LAN_100:
			d1printf("100Mb/s Voice Grade AnyLAN network.\n" );
			return(B_NO_ERROR);
			break;
		case HP100_LAN_10:
			d1printf("10Mb/s network.\n" );
			return(B_NO_ERROR);
			break;
		default:
			d1printf("Warning! Link down.\n" );
	}
	hp100_stop_interface( data ); 
	return(B_ERROR);
}

/* 
 * Load EEPROM defaults into registers - quite slow
 */
static void 
hp100_load_eeprom( hp_private_t *data )
{
	int i;
	int ioaddr = data -> base_addr;

	hp100_page( EEPROM_CTRL );
	/* initiate EEPROM load into regs */
	hp100_andw( ~HP100_EEPROM_LOAD, EEPROM_CTRL );
	hp100_orw( HP100_EEPROM_LOAD, EEPROM_CTRL );
	for ( i = 0; i < 6000; i++ )
		/* HP100_EE_LOAD resets when load is complete... */
		if ( !( hp100_inw( OPTION_MSW ) & HP100_EE_LOAD ) ) return;
	d0printf( "hp100_load_eeprom: - timeout\n" );
}

/*
 * Initialize the hardware into a safe state, including a full PCI fifo reset
 * Makes many assumptions about desired operating mode etc.
 */
static void 
hp100_hwinit(hp_private_t *data) 
{
	int ioaddr=data->base_addr;
	int i;
	
	/* clear interrupts etc... */
	hp100_page( PERFORMANCE );
	hp100_outw( 0xfefe, IRQ_MASK );			/* mask all interrupts off */
	hp100_outw( 0xffff, IRQ_STATUS );		/* ack pending IRQs */
	hp100_outw( HP100_INT_EN | HP100_RESET_LB, OPTION_LSW );
	hp100_page( MAC_CTRL );
	/* turn off receive/transmit functions - no need to wait, we don't care */
	hp100_andb( ~(HP100_RX_EN|HP100_TX_EN), MAC_CFG_1 );
	/* do full reset, including PCI fifo */
	/* eventually this stuff should be in a generic cascade reset function... */
	hp100_outw( HP100_HW_RST | HP100_RESET_LB, OPTION_LSW );
	hp100_page ( HW_MAP );
	/* turn on ISR clear-on-read - disallows ISR debugging but makes 
	 * read/clear atomic (read: very handy) */
	hp100_orb( HP100_ISR_CLRMODE, MODECTRL1 );
	hp100_outb( 0x00, BM );
	hp100_outb( 0, EARLYRXCFG );
	hp100_outw( 0, EARLYTXCFG );
	hp100_andb( ~HP100_PCI_RESET, PCICTRL2 );
	hp100_orb( HP100_PCI_RESET, PCICTRL2 );
	for(i=0; i<0xffff; i++);
	hp100_andb( ~HP100_PCI_RESET, PCICTRL2 );
	hp100_page( PERFORMANCE );
	/* load factory defaults */
	hp100_load_eeprom( data );
	/* set reasonable defaults for a basic PIO driver */
	hp100_outw( HP100_MMAP_DIS | HP100_SET_HB | HP100_IO_EN | HP100_SET_LB, OPTION_LSW );
	hp100_outw( HP100_DEBUG_EN | HP100_RX_HDR | HP100_EE_EN | 
				HP100_BM_WRITE | HP100_BM_READ | HP100_RESET_HB |
				HP100_FAKE_INT | HP100_MEM_EN | HP100_RESET_LB, OPTION_LSW );
	hp100_outw( HP100_ADV_NXT_PKT | HP100_TX_CMD | HP100_RESET_LB |
				HP100_PRIORITY_TX | HP100_RESET_HB, OPTION_MSW );
	/* bring out of reset */
	hp100_outw( HP100_HW_RST|HP100_SET_LB, OPTION_LSW );
	for(i=0; i<0xffff; i++);
	hp100_page( PERFORMANCE );
	/* attempt hub login */
	if ( data->lan_type != HP100_LAN_10 )
		hp100_login_to_vg_hub( data );
}

/* 
 *  Init entry point - calls probes, creates semaphores, and configures card
 */
static int 
hp100_init(device_info *drvr) 
{
	hp_private_t *data;
	int ioaddr;
	int i,res;
	
	data=&hp100_data;
	if (hp100_probe(data) != B_NO_ERROR) {
		data->port = 0;
		d0printf("HP100 probe failed\n");
		return (B_ERROR);
	}
	ioaddr=data->base_addr;
	/* locks hardware access - begins unlocked */
	data->iolock = create_sem(1, "hp100 io");
	set_sem_owner(data->iolock, B_SYSTEM_TEAM);
	/* blocks writers - begins unlocked */
	data->olock = create_sem(1, "hp100 output");
	set_sem_owner(data->olock, B_SYSTEM_TEAM);
	/* blocks readers - begins locked, unlocked on packet arrival */
	data->ilock = create_sem(0, "hp100 input");
	set_sem_owner(data->ilock, B_SYSTEM_TEAM);
	data->interrupted = 0;
	data->inrw = 0;
	data->inint = 0;
	data->rxunack = 0;
	/* hp100 specific init things */
	hp100_page( MAC_CTRL );
	/* enable 10Mbit link (used only for 10Mbit MAC) */
	hp100_orb( HP100_LINK_BEAT_DIS | HP100_RESET_LB, 10_LAN_CFG_1 );
				
	hp100_page( MAC_ADDRESS );
	/* load real, hardware-assigned MAC address into page 1 */
	for (i = 0; i < 6; i++) 
		hp100_outb( data->dev_addr.ebyte[ i ], MAC_ADDR + i );
	/* clear multicast list */
	for ( i = 0; i < 8; i++)
		hp100_outb ( 0x00, HASH_BYTE0 + i );
	/* set hardware to a safe state */
	hp100_hwinit( data );
	hp100_page( PERFORMANCE );
	/* set IRQ mask to interrupts that we want enabled */
	hp100_outw( (INTS_WE_CARE_ABOUT | HP100_SET_HB | HP100_SET_LB), IRQ_MASK );
	d5printf("hp100: set mask to 0x%04x\n",hp100_inw( IRQ_MASK ));

	/* divide card's memory buffer into TX and RX rings, based on rx_ratio */
	hp100_page( MMU_CFG );
    hp100_outw( (((data->memory_size*data->rx_ratio)/100)>>4), RX_MEM_STOP );
    hp100_outw( ((data->memory_size - 1 )>>4), TX_MEM_STOP );  

	hp100_page( MAC_CTRL );
	/* workaround bug involving 802.12 cards and old hubs (like mine) */
	hp100_outb( 0x00, VG_LAN_CFG_2 );
	/* get that baby running again */	
	hp100_start_interface( data );
	/* login to hub, if we must */
	if (data->lan_type == HP100_LAN_100)
		data->hub_status = hp100_login_to_vg_hub( data );
	if (data->hub_status == B_NO_ERROR)	
		d1printf("hp100: successfully logged into hub\n");
	else {
		d0printf("hp100: login to hub unsuccessful\n");
		return(B_ERROR);
	}	
	/* install our interrupt handler on the top of the I/O bus */
	res=set_io_interrupt_handler(data->irq_level, hp100_interrupt, (void *)data);
	if (res != B_NO_ERROR) 
		d0printf("hp100: unable to install interrupt handler! \n");
	else
		d1printf("hp100: installed intr handler at IRQ 0x%02x\n",data->irq_level);
	d5printf("hp100: enabling intr: option_lsw: 0x%04x\n",hp100_inw( OPTION_LSW ));	
	/* turn the interrupt on  */
	res=enable_io_interrupt(data->irq_level);
	if (res == B_ERROR)
		d0printf("hp100: unable to enable IO interrupt\n");
	else
		d1printf("hp100: IO interrupt enabled\n");
	drvr->private_data = data;
	/* change page to PERFORMANCE for run-time operation! full speed ahead! */
	hp100_page( PERFORMANCE );
	/* whoo-hoo!  Everything checks out... */
	return (B_NO_ERROR);
}

/* 
 * 'Enables' the card, by turning on interrupts and turning on RX and TX functions
 * should lock the card, although this stuff shouldn't really be happening live
 */
static void 
hp100_start_interface(hp_private_t *data)
{
	int ioaddr = data -> base_addr;

	hp100_page( MAC_CTRL );
	/* set the MAC modes (promiscuous etc...)
	   in 100VG, the hub has final say - check to see if it was trained ? */
	hp100_outb( data -> mac2_mode, MAC_CFG_2 );
	hp100_andb( HP100_MAC1MODEMASK, MAC_CFG_1 );
	/* hack - should set this properly? */
	hp100_orb( data -> mac1_mode |
			   HP100_RX_EN | HP100_RX_IDLE |
			   HP100_TX_EN | HP100_TX_IDLE, MAC_CFG_1 );
	hp100_page( PERFORMANCE );
	hp100_outw( HP100_TRI_INT | HP100_RESET_HB, OPTION_LSW );
	hp100_outw( HP100_INT_EN | HP100_SET_LB, OPTION_LSW ); 
} 

/* 
 * 'Disables' the card, by turning off interrupts and stopping RX and TX 
 * functions, pending RX/TX's go to completion, and we wait
 */
static void 
hp100_stop_interface(hp_private_t *data)
{
	int ioaddr = data -> base_addr;
	u_short val;

	hp100_page( PERFORMANCE );
	/* turn off interrupts */
	hp100_outw( HP100_INT_EN | HP100_RESET_LB | 
				HP100_TRI_INT | HP100_SET_HB, OPTION_LSW );
	val = hp100_inw( OPTION_LSW );
	hp100_page( MAC_CTRL );
	/* turn off transmit/receive functions ... */
	hp100_andb( ~(HP100_RX_EN | HP100_TX_EN), MAC_CFG_1 );
	/* if in reset, that's perfect */
	if ( !(val & HP100_HW_RST) ) return;
	/* spin until idle */
	for ( val = 0; val < 6000; val++ )
		if ( ( hp100_inb( MAC_CFG_1 ) & (HP100_TX_IDLE | HP100_RX_IDLE) ) ==
										(HP100_TX_IDLE | HP100_RX_IDLE) )
		return;
	d0printf( "hp100: hp100_stop_interface - timeout\n");
}

/*
 * Copy a packet from the ethernet card (using Programmed I/O)
 */
static int
copy_packet(
			hp_private_t *data,
			unsigned char *hp100_buf,
			int buflen
			)
{
	int ioaddr = data->base_addr;
	int pkt_len=0,i;
	unsigned int header;

	/* doing hardware access -- lock! */
	io_lock(data);
	d3printf("hp100: in copy packet: we have %d packets left\n",data->rxunack);
	/* read a packet only if there are any left */
	/* recent tightening of the code should make this condition always
	 * true, but it's safer this way*/
	if (data->rxunack > 0) {
		/* if we haven't finished advancing from last packet, wait */
		for (i=0;i<6000 && (hp100_inw(OPTION_MSW)&HP100_ADV_NXT_PKT);i++) ;
		/* get packet header -- typecast to something more interesting? */
		/* endianness? */
		if (!(hp100_inb(RX_PKT_CNT) > 0)) {
			d0printf("hp100: ghost packet, card says we have %d waiting\n",
					 hp100_inb(RX_PKT_CNT));
			return(0);
		} 
		header = SWAPLONG(hp100_inl( DATA32 ));
		d5printf("hp100: copy packet: raw header = 0x%04x\n",header);
		pkt_len = header & HP100_PKT_LEN_MASK;
		/* mungy stuff - need a nice bit map or something? */
		d5printf("hp100_rx: new packet - length = %d, errors = 0x%x, dest = 0x%x\n",
			pkt_len, (header >> 16) & 0xfff8, (header >> 16) & 7 );
		/* use PIO to grab data from card - buffer had better be aligned to
		 * the next long-word! */
		hp100_insl(data, HP100_REG_DATA32, (unsigned long *)hp100_buf, 
				   (pkt_len+3)>>2);
		/* should do stats update and error checking in here */
		/* advance the card's pointer to the next packet */
		hp100_outw( HP100_ADV_NXT_PKT | HP100_SET_LB, OPTION_MSW );
		atomic_add(&data->rxunack, -1);
	}
	d3printf("hp100: leaving copy packet: we have %d packets left\n",
		     data->rxunack);
	/* done hardware access -- unlock */
	io_unlock(data);
	return (pkt_len);
}

/* 
 * Entry point for hardware interrupts - eventually, we will need to allow for
 * shared interrupts, but support isn't there yet.  This function must be /fast/.
 */
static bool
hp100_interrupt(void *vdata)
{
	hp_private_t *data=(hp_private_t *)vdata;
	ushort isr=0,pkt_cnt;
	int ioaddr = data->base_addr;
	int wakeup_reader = 0;
	int wakeup_writer = 0;

	/* 'inint' is largely an artifact, sometimes useful */
	atomic_add(&data->inint,1);
	data->ints++;
	/* lockingly grab and clear the ISR */
	INTR_LOCK(data, isr = hp100_inw( IRQ_STATUS )); 
	if (isr & HP100_RX_PACKET) {
		pkt_cnt=hp100_inb(RX_PKT_CNT);
		if (pkt_cnt > data->rxunack) {
			d3printf("hp100: rx interrupt caught - waking reader - %d new packets\n",
					 wakeup_reader=(pkt_cnt - data->rxunack));
			atomic_add(&data->rxunack, wakeup_reader);
		}
	}
	if (isr & HP100_TX_COMPLETE) {
		d2printf("hp100: tx complete interrupt caught - waking writer\n");
		wakeup_writer++; 
	}
	/* get set for a retransmit or whatever on error */
    if ( isr & ( HP100_TX_ERROR | HP100_RX_ERROR ) ) {
		if (isr & HP100_TX_ERROR) 
			wakeup_writer++;
		if (isr & HP100_RX_ERROR) 
    		d0printf("hp100: receive error, stats incomplete\n");
	}
	/* unblock threads as the ISR indicated */
  	if (wakeup_reader) {
		input_unwait(data, wakeup_reader);
 	}
    if (wakeup_writer) {
 		output_unwait(data, 1);
	}
	atomic_add(&data->inint,-1);
	/* return TRUE to indicate we dealt with this interrupt */
	return TRUE;
}

/*
 * exported function to read from the device
 * doesn't do any real work
 */
#pragma export on
static long 
hp100_read(device_info *info, void *buf, ulong buflen, ulong pos) 
{
	int packet_len;
	hp_private_t *data = (hp_private_t *)info->private_data;

	if (data == NULL) {
		return (B_ERROR);
	}
	atomic_add(&data->inrw, 1);
	/* check the 'interrupted' flag to see if we were forcibly unblocked */
	if (data->interrupted) {
		atomic_add(&data->inrw, -1);
		return (B_INTERRUPTED);
	}
	/* loop until we can return a good, long packet */
	do {
		d3printf("hp100: waiting for packet rx, sem count = %d, unack = %d\n",
				 input_count(data), data->rxunack); 
		/* block calling thread on data */
		input_wait(data);
		d3printf("hp100: read unblocked, sem count = %d, unack = %d\n",
				 input_count(data), data->rxunack);
		if (data->interrupted) {
			atomic_add(&data->inrw, -1);
			return (B_INTERRUPTED);
		}
		packet_len = copy_packet(data, (unsigned char *)buf, buflen);
	} while (packet_len == 0);
	d3printf("hp100: rx packet, len = %d\n",packet_len);
	atomic_add(&data->inrw, -1);
	return (packet_len);
}
#pragma export off

/*
 * exported function to write a packet
 * this (apparently) works! whoo-hoo!
 */
#pragma export on
static long 
hp100_write(device_info *info, void *buf, ulong buflen, ulong pos) 
{
	hp_private_t *data = (hp_private_t *)info->private_data;
	int status,i,ok_flag;
	int ioaddr=data->base_addr;
	sem_info semtest;
	
	if (data == NULL) {
		return (B_ERROR);
	}
	
	atomic_add(&data->inrw, 1);
	if (data->interrupted) {
		atomic_add(&data->inrw, -1);
		return (B_INTERRUPTED);
	}
	/*
	 * Wait for somebody else (if any) to finish transmitting 
	 */
	get_sem_info(data->olock,&semtest);
	d2printf("hp100: waiting, olock sem count = %d lsw = 0x%04x\n",
			 semtest.count, hp100_inw( OPTION_LSW ));
	status = output_wait(data, HP100_TRANSMIT_TIMEOUT);
	if (status < B_NO_ERROR || data->interrupted) {
		atomic_add(&data->inrw, -1);
		return (status);
	}
	/* wait for hardware lock */
	io_lock(data);
	/* for some reason, things are haywire... fix things up! */
	if (data -> lan_type < 0) {
		hp100_stop_interface( data );
		if ( ( data->lan_type = hp100_sense_lan( data ) ) < 0 )
		{
			d0printf("hp100: no connection found - check wire\n");
			hp100_start_interface( data );
			return(B_ERROR);
		}
		if (data->lan_type == HP100_LAN_100)
			data->hub_status = hp100_login_to_vg_hub( data );
		hp100_start_interface( data );
	} 

	/* mungy memory check */
	/* check the logic of these hub logins etc... */
	if ((i = (hp100_inl( TX_MEM_FREE ) & 0x7fffffff )) < (buflen+16)) {
		d0printf("hp100: possible endian problem - TX_MEM_FREE = 0x%x\n",
				 hp100_inl( TX_MEM_FREE ));
		d0printf("hp100: write, free mem = 0x%x\n",i);
		if ((data->lan_type == HP100_LAN_100) && (data->hub_status < 0)) { 
			d0printf("hp100: login to 100Mb/s hub retry\n");
			hp100_stop_interface( data );
			data->hub_status = hp100_login_to_vg_hub( data );
			hp100_start_interface( data );
		}
		else {
			hp100_ints_off();
			i=hp100_sense_lan( data );
			hp100_page( PERFORMANCE );
			hp100_ints_on();
			if (i==HP100_LAN_ERR) 
				d0printf("hp100: link down detected\n");
			else if (data->lan_type != i) {  /* cable swap !!! */
				d0printf("hp100: cable change 10Mb/s <--> 100Mb/s detected\n");
				data->lan_type=i;	
				hp100_stop_interface( data );
				if (data->lan_type == HP100_LAN_100)
					data->hub_status = hp100_login_to_vg_hub( data );
				hp100_start_interface( data );
			}
			else {
				d0printf("hp100: interface reset\n");
				hp100_stop_interface( data );
				hp100_start_interface( data );
			}
		}
		return(B_ERROR);
	}
	/* pad runt packets to spec length */
	ok_flag = (buflen >= HP100_MIN_PACKET_SIZE);
	i = ok_flag ? buflen : HP100_MIN_PACKET_SIZE;
	buflen = i;
	hp100_outw( i, DATA32 );
	hp100_outw( i, FRAGMENT_LEN );
	/* use PIO to feed data into card, long-aligned! */
	hp100_outsl(data, HP100_REG_DATA32, (const unsigned long *)buf, 
				(buflen + 3) >> 2);
	d2printf("hp100: packet uploaded through\n");
	/* pad packet with 0's (long aligned!) */
	if (!ok_flag)
		for (i = (buflen + 3) & ~3; i < HP100_MIN_PACKET_SIZE; i+=4) 
			hp100_outl( 0, DATA32 );
	hp100_outw( HP100_TX_CMD | HP100_SET_LB, OPTION_MSW ); 
	d5printf("hp100: optionlsw: 0x%04x \n",
			 (unsigned short)hp100_inw( OPTION_LSW ));
	data->writes++;
	io_unlock(data);
	atomic_add(&data->inrw, -1);
	d5printf("hp100: write finished \n");
	return (buflen);
}
#pragma export off

/*
 * very cursory exported open function, as per Be's example 
 * the real work is done in hp100_control 
 */ 
#pragma export on
static long 
hp100_open(device_info *info, ulong flags) 
{
	info->private_data=NULL;
	set_dprintf_enabled(TRUE);
	return(B_NO_ERROR);
}
#pragma export off

/* 
 * simple exported close function - clears semaphores etc.
 */
#pragma export on
static long 
hp100_close(device_info *drvr) 
{
	hp_private_t *data=(hp_private_t *)drvr->private_data;

	if (data == NULL) {
		/*
		 * We didn't find a card: No need to do anything
		 */
		return (B_NO_ERROR);
	}
	/* 
	 * Force pending reads and writes to terminate
	 */
	io_lock(data);
	/* here we notify them that something bad has interrupted them */
	data->interrupted = 1;
	input_unwait(data, 1);
	output_unwait(data, 1);
	io_unlock(data);
	/* let current reads/writes finish properly */
	while (inrw(data)) {
		snooze(1000000);
		d1printf("hp100: still waiting for read/write to finish\n");
	}

	/*
	 * Stop the chip
	 */
	hp100_stop_interface( data );
	snooze(2000.0);
	/*
	 * And clean up
	 */
	disable_io_interrupt(data->irq_level);
	delete_sem(data->iolock);
	delete_sem(data->ilock);
	delete_sem(data->olock);
	set_io_interrupt_handler(data->irq_level, NULL, NULL);
	/*
	 * Reset all the statistics 
	 */
	data->ints = 0;	
	data->rints = 0;	
	data->rerrs = 0;	
	data->wints = 0;	
	data->werrs = 0;	
	data->reads = 0;	
	data->writes = 0;	
	data->interrs = 0;
	data->resets = 0;
	data->frame_errs = 0;
	data->crc_errs = 0;
	data->frames_lost = 0;
	
	data->rerrs_last = 0;	
	data->werrs_last = 0;	
	data->interrs_last = 0;
	data->frame_errs_last = 0;
	data->crc_errs_last = 0;
	data->frames_lost_last = 0;
	/*
	 * Reset to notify others that this port and irq is available again
	 * (whatever that means)
	 */
	data->port = 0;
	data->irq_level = 0;
	return (B_NO_ERROR);
}
#pragma export off

/*
 * the real exported open/init function 
 */
#pragma export on
static long 
hp100_control(device_info *info, ulong op, void *buff) 
{
	hp_private_t *data;

	switch (op) {
		/* init the card */
		case ETHER_INIT:
			return (hp100_init(info)); 
		case ETHER_GETADDR:
		/* get the card's MAC address */
			data = (hp_private_t*)info->private_data;
			if (data == NULL)
				return(B_ERROR);
			memcpy(buff, &data->dev_addr,sizeof(data->dev_addr)); 
			return(B_NO_ERROR);
	}
	return B_ERROR;
}
#pragma export off

/* 
 * our exported device table entry, with device name etc...
 */
#pragma export on
device_entry devices[] = {
	{
		"/dev/hp100",
		hp100_open,
		hp100_close,
		hp100_control,
		hp100_read,
		hp100_write
	},
	0
};

#pragma export off
